#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifndef UNITTEST
#include <sys/neutrino.h>
#include "input/mtouch_driver.h"
#include <input/mtouch_log.h>
#else
#include "touchscreen_header.h"
#include "interrupt_header.h"
#include "stdio.h"
#endif

#include "touch.h"

int
cypress_read_bootloader_reply(cypress_dev_t* cypress, uint8_t **reply);

uint16_t
calculate_bootloader_crc(uint8_t *buf, int len)
{
    uint16_t crc = 0xffff;
    uint16_t tmp;
    int i;

    if (len == 0)
        crc = ~crc;
    else {
        do {
            for (i = 0, tmp = 0x00ff & *buf++; i < 8; i++, tmp >>= 1) {
                if ((crc & 0x0001) ^ (tmp & 0x0001))
                    crc = (crc >> 1) ^ 0x8408;
                else
                    crc >>= 1;
            }
        } while (--len);

        crc = ~crc;
        tmp = crc;
        crc = (crc << 8) | (tmp >> 8 & 0xFF);
    }

    return crc;
}

/* TODO: ERICK Combine the two cksum functions */
uint8_t
quick_cksum (uint8_t *data, uint8_t len)
{
  uint16_t cksum = 0;
  int i;

  for (i = 0; i < len; i++)
    cksum += data[i];

  return (~(cksum & 0xFF) + 1);
}

uint8_t
calculate_cksum (struct cyacd_record *record, uint8_t *validate)
{
    uint16_t cksum = 0;
    int i;

    cksum += record->array_id;
    cksum += record->row_number >> 8;
    cksum += record->row_number & LSB;
    cksum += record->record_length >> 8;
    cksum += record->record_length & LSB;

    for (i = 0; i < record->record_length; i++)
        cksum += record->data[i];

    if (validate != NULL)
        cksum += *validate;

    return (~(cksum & 0xFF) + 1);
}

int
cleanup(cypress_dev_t* cypress)
{
    struct cyacd_record *curr, *tmp;

    curr = cypress->head_record;

    while(curr) {
        if (curr->data != NULL)
            free (curr->data);

        tmp = curr->next;
        free (curr);

        curr = tmp;
    }

    return EOK;
}

/* Clear the error pin and attempt to recover the bootloader */
int
cypress_bootloader_clear_error_pin(cypress_dev_t* cypress)
{
        uint8_t *reply = NULL;
    int reply_len = 0;
    static int waiting_on_reply = 0;

    if (!waiting_on_reply) {
        if ((cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_CLEAR_ERRORS, 0, NULL, 0)) < 0) {
            mtouch_error (cypress->log_name, "Bootloader Clear Errors failed");
            return -1;
        }

        waiting_on_reply = 1;
    } else {
        /* We were waiting for a reply from the controller */
        if ((reply_len = cypress_read_bootloader_reply (cypress, &reply)) == -1) {
            mtouch_error (cypress->log_name, "%s: Failed to get reply from bootloader", __FUNCTION__);
            error_memory ("Cypress_Touch: %s: Failed to get reply from bootloader", __FUNCTION__);
            return -1;
        }

        waiting_on_reply = 0;

#if 0
        /* The reply to this command is not documented well, it simply states the status as "variable" not detailing what this might be. */
        if (reply[1])
            mtouch_info (cypress->log_name, "Clear Errors successful");
        else
            mtouch_error (cypress->log_name, "Failed to clear error pin");
#endif

        if (cypress->error_pin_status & ERROR_APPLICATION_CRC) {
            /* A firmware update is required */
            cypress->bootloader_state = WAITING_UPDATE;
        } else {
            /* We are not waiting on a firmware update and this is a regular recoverable error */
            if (EOK != cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_EXIT_CRC, 0, NULL, 0)) {
                free (reply);
                return -1;
            }

            cypress->bootloader_state = APPLICATION_RUNNING;
            cypress->driver_state = SYSTEM_INFORMATION;
            cypress_update_pps_status(cypress, "status::system_information");

            /* Start watchdog, just in case controller gets stuck */
            cypress_set_boot_watchdog_timer(cypress, 0);
        }

        free (reply);
    }

    return EOK;
}

int
cypress_set_boot_watchdog_timer(cypress_dev_t* cypress, unsigned sec)
{
    int rc;

    cypress->itime.it_value.tv_sec = sec;
    cypress->itime.it_value.tv_nsec = 0;
    cypress->itime.it_interval.tv_sec = 0;
    cypress->itime.it_interval.tv_nsec = 0;

    /* Set heartbeat timer */
    rc = timer_settime(cypress->boot_watchdog_timerid, 0, &(cypress->itime), NULL);

    if (EOK != rc) {
        mtouch_error (cypress->log_name, "Failed to set the boot watchdog timer to %d.  Error: %d", sec, errno);
        error_memory ("Cypress_Touch: Failed to set the boot watchdog timer to %d.  Error: %d", sec, errno);
        return -1;
    }

    return EOK;
}

int
cypress_decypher_family_gen(cypress_dev_t* cypress, uint8_t buf)
{
    switch (buf) {
    case GEN_FOUR_ID:
        return GEN_FOUR;
    case GEN_SIX_ID_CYTAT81X:
    case GEN_SIX_ID:
        return GEN_SIX;
    default:
        mtouch_warn (cypress->log_name, "Unrecognized silicon generation: %d", buf);
    }

    return -1;
}

int
cypress_bootloader_display_packet(cypress_dev_t* cypress, uint8_t *packet, int len, int operation)
{
    char *buf;
    int i;

    if ((len*3) > (MAX_SLOG_BUFFER*4)) {
        mtouch_error (cypress->log_name, "Packet to be displayed is larger than allowable limit (255*4)");
        return 0;
    }

    buf = (char *) malloc (len*5);  /* 5 times the size of the buffer to accomodate for the converted chars from hex values */

    if (buf == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate buffer space to dump touch record");
        return -1;
    }

    sprintf (buf, "%s %02hhX ", operation ? "r":"w", packet[0]);

    for (i = 1; i < len; i++) {
        sprintf ((buf + strlen(buf)), "%02hhX ", packet[i]);
    }

    mtouch_info (cypress->log_name, "%s", buf);

    free (buf);

    return 0;
}

void reverse_list(struct cyacd_record **head)
{
    struct cyacd_record *prev = NULL, *next = NULL, *curr = *head;

    while(curr!=NULL){
        next = curr->next;
        curr->next = prev;
        prev = curr;
        curr = next;
    }
    *head = prev;
}

int
cypress_bootloader_open_file(cypress_dev_t* cypress)
{
    char buf[MAX_LINE_SIZE] = { 0 };
    FILE *fp;
    struct cyacd_record *curr;
    int i;
    uint8_t cksum = 0;
    int cksum_failure = 0;
    char *tmp;
	unsigned int buf_size = 0;

    cypress->head_record = NULL;
    cypress->firmware_line_count = 0;

    /* Open file */
    if (NULL == cypress->firmware_file) {
        mtouch_error (cypress->log_name, "No firmware file specified, cannot update firmware");
        return -1;
    }

    if ((tmp = strdup (cypress->firmware_file)) == NULL) {
        mtouch_error (cypress->log_name, "String dup failed on firmware filename");
        error_memory ("Cypress_Touch: String dup failed on firmware filename");
        return -1;
    }

    if ((strstr ((strlwr(tmp)), ".cyacd")) == NULL) {
        mtouch_error (cypress->log_name, "Firmware must be a .cyacd text file");
		free(tmp);
        return -1;
    }

    free(tmp);

    if (cypress->verbose)
        mtouch_info (cypress->log_name, "Opening %s", cypress->firmware_file);

    /* Check to see if we can open the file */
    // if (!access(cypress->firmware_file, R_OK)) {
    fp = fopen (cypress->firmware_file, "r");
    if (NULL == fp) {
        mtouch_error (cypress->log_name, "Error opening firmware file %s: %d", cypress->firmware_file, errno);
        return -1;
    }

    /* Read in all the records */
    while(fgets(buf, sizeof(buf), fp)!= NULL) {

        /* Check to see if this is the header or a line */
        if (buf[0] == ':') {
            curr = (struct cyacd_record *)malloc (sizeof (struct cyacd_record));

            if (curr == NULL) {
                mtouch_error (cypress->log_name, "Failed to malloc memory for record entry");
                fclose (fp);
                return -1;
            }

            curr->array_id = ((HEX(buf[1])) << 4); /* Array id is uint8_t */
            curr->array_id |= HEX(buf[2]);
            curr->row_number = ((HEX(buf[3])) << 12); /* Row number is a uint16_t */
            curr->row_number |= ((HEX(buf[4])) << 8);
            curr->row_number |= ((HEX(buf[5])) << 4);
            curr->row_number |= HEX(buf[6]);
            curr->record_length = ((HEX(buf[7])) << 12); /* Record length is a uint16_t */
            curr->record_length |= ((HEX(buf[8])) << 8);
            curr->record_length |= ((HEX(buf[9])) << 4);
            curr->record_length |= HEX(buf[10]);

			buf_size = sizeof(buf) - CYACD_DATA_OFFSET;
			curr->record_length = curr->record_length > buf_size/4 ?  buf_size/4:curr->record_length;
            curr->data = (uint8_t *)calloc(curr->record_length, sizeof(uint8_t));

            if (curr->data == NULL) {
                mtouch_error (cypress->log_name, "Failed to allocate memory for record");
                free (curr);
                fclose (fp);
                return -1;
            }

            /* Convert text string from CYACD file to HEX and put it into a uint8_t array */
            /* We multiply i by 2 as each character in the file only account for 1/2 the uint8_t value */

            for (i = 0; i < curr->record_length; i++) {
                curr->data[i] = (((HEX(buf[CYACD_DATA_OFFSET + (i * 2)])) << 4) & 0xF0); /* High 4 bits */
                curr->data[i] |= ((HEX(buf[CYACD_DATA_OFFSET + (i * 2) + 1])) & 0x0F);   /* Low 4 bits */
            }

            curr->cksum = (((HEX(buf[CYACD_DATA_OFFSET + (curr->record_length * 2)])) << 4) & 0xF0); /* High 4 bits */
            curr->cksum |= HEX(buf[CYACD_DATA_OFFSET + (curr->record_length * 2) + 1]);              /* Low 4 bits */

            cksum = calculate_cksum(curr, NULL);

            if (curr->cksum != cksum) {
                mtouch_error (cypress->log_name, "Error, checksum for array ID %d row %d does not match! Found %x Expected %x\n", curr->array_id, curr->row_number, curr->cksum, cksum);
                error_memory ("Cypress_Touch: Error, checksum for array ID %d row %d does not match! Found %x Expected %x\n", curr->array_id, curr->row_number, curr->cksum, cksum);

                /* Check to make sure we calculated things correctly */
                if (calculate_cksum(curr,&cksum))
                    mtouch_error (cypress->log_name, "CKSUM was calculated incorrectly\n");
                else
                    mtouch_error (cypress->log_name, "CKSUM in file is incorrect!\n");

                cksum_failure = 1;
            }

            curr->next = cypress->head_record;
            cypress->head_record = curr;
            cypress->firmware_line_count++;
        } else {
            /* Header */
            cypress->file_chip_revision = (((HEX(buf[CHIP_REVISION])) << 4) | (HEX(buf[((CHIP_REVISION) + 1)])));
            cypress->file_silicon_id = (((HEX(buf[SILICON_ID])) << 12) | ((HEX(buf[((SILICON_ID) + 1)])) << 8) | ((HEX(buf[((SILICON_ID) + 2)])) << 4) | (HEX(buf[((SILICON_ID) + 3)])));
        }
    }


    if (cksum_failure) {
        cleanup(cypress);
        fclose (fp);
        return -1;
    }

    /* Reverse the list */
    reverse_list(&cypress->head_record);

    if (cypress->verbose)
        mtouch_info (cypress->log_name, "%s opened and read succesfully", cypress->firmware_file);

    fclose (fp);

    return 0;
}

int
cypress_parse_bootlaoder_error(cypress_dev_t* cypress, uint8_t *reply)
{
    int status;

    status = reply[1];

    switch (status) {
    case SUCCESS:
        if (cypress->verbose > 4)
            mtouch_info (cypress->log_name, "Command was successfully received and executed.");
        break;
    case INCORRECT_COMMAND:
        mtouch_warn (cypress->log_name, "The 8-byte key string in the 'initiate Bootloader' command is incorrect.");
        break;
    case FLASH_VERIFICATION_FAILED:
        mtouch_warn (cypress->log_name, "The flash verification failed.");
        break;
    case DATA_SIZE_OUT_OF_RANGE:
        mtouch_warn (cypress->log_name, "The amount of data available is outside the expected range.");
        break;
    case DATA_IN_INCORRECT_FORM:
        mtouch_warn (cypress->log_name, "The data is not of proper form");
        break;
    case UNRECOGNIZED_COMMAND:
        mtouch_warn (cypress->log_name, "The command is not recognized, or was issued in the wrong state.");
        break;
    case CRC_DOES_NOT_MATCH:
        mtouch_warn (cypress->log_name, "The communication CRC does not match the expected value.");
        break;
    case INVALID_FLASH_ROW:
        mtouch_warn (cypress->log_name, "Invalid flash row.");
        break;
    case FLASH_ROW_WRITE_PROTECTED:
        mtouch_warn (cypress->log_name, "Flash row write protected.");
        break;
    case 0x06:
    case 0x07:
    case 0x0C:
    case 0x0D:
    case 0x0E:
    case 0x0F:
        /* Reserved */
        break;
    default:
        mtouch_error (cypress->log_name, "Unknown status error from Bootloader! %x", status);
        error_memory ("Cypress_Touch: Unknown status error from Bootloader! %x", status);
        break;
    }

    return status;
}

int
cypress_read_bootloader_reply(cypress_dev_t* cypress, uint8_t **reply)
{
    uint8_t *buf = 0;
    uint16_t crc = 0x0;
    uint16_t sent_crc = 0x0;
    int data_len = 0;
    int packet_size = 0;

    buf = (uint8_t *)calloc(RETURN_PACKET_HEADER_LEN + RETURN_PACKET_TAIL_LEN, sizeof(uint8_t));
    if (buf == NULL) {
        mtouch_error(cypress->log_name, "Failed to allocate memory for bootloader reply");
        return -1;
    }

    data_len = ((buf[RETURN_PACKET_DATA_LENGTH_MSB] << 8) | buf[RETURN_PACKET_DATA_LENGTH_LSB]);
    packet_size = (RETURN_PACKET_HEADER_LEN + data_len + RETURN_PACKET_TAIL_LEN);

    buf = (uint8_t *)realloc(buf, packet_size * sizeof(uint8_t));
    if (buf == NULL) {
        mtouch_error(cypress->log_name, "Failed to reallocate memory for bootloader reply");
        return -1;
    }

    /* Read the rest of the reply packet */
    if ((cypress->i2c_funcs.read_reg (cypress->i2c_fd, HST_MODE, packet_size, buf)) < 0)
    {
         mtouch_error(cypress->log_name, " failed to read full %d size packet", packet_size);
         return -1;
    }

    if (buf[(packet_size - 1)] != CYPRESS_BOOTLOADER_PACKET_END) {
        mtouch_error (cypress->log_name, "End of Return Packet not found!!");
        error_memory ("Cypress_Touch: End of Return Packet not found!!");
        cypress_bootloader_display_packet(cypress, buf, packet_size, READ);
        return -1;
    }

    /* We have a complete packet, check CRC */
    sent_crc = buf[RETURN_PACKET_HEADER_LEN + data_len + 1] << 8;
    sent_crc |= buf[RETURN_PACKET_HEADER_LEN + data_len];

    /* Check CRC, remove the end byte and CRC */
    crc = calculate_bootloader_crc(&(buf[CYPRESS_BOOTLOADER_PACKET_START_OFFSET]), (RETURN_PACKET_HEADER_LEN + data_len));

    if (crc != sent_crc) {
        /* CRC did not match! */
        mtouch_error (cypress->log_name, "Return Packet CRC failed! CRC Found: %x Expected %x", sent_crc, crc);
        error_memory ("Cypress_Touch: Return Packet CRC failed! CRC Found: %x Expected %x", sent_crc, crc);
        cypress_bootloader_display_packet(cypress, buf, packet_size, READ);
        return -1;
    }

    if (*reply != NULL)
        free (*reply);

    *reply = (uint8_t *)calloc(packet_size, sizeof (uint8_t));

    if (*reply == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate memory for bootloader data response");
        return -1;
    }

    memcpy (*reply, buf, packet_size);

    if (cypress->verbose > 5)
        cypress_bootloader_display_packet(cypress, *reply, packet_size, READ);

    free(buf);
    return (packet_size);
}


int
cypress_bootloader_cmd(cypress_dev_t* cypress, uint8_t cmd, uint8_t addr, uint8_t *data, uint8_t data_len)
{
    int len = 8; /* Minimum 8 bytes */
    int rc = 0;
    uint16_t crc = 0x0;
    uint8_t *buf;
    int i;

    len += data_len;
    buf = (uint8_t *)calloc(len, sizeof (uint8_t));

    if (buf == NULL) {
        mtouch_error (cypress->log_name, "Failed to calloc space for bootloader command");
        return -1;
    }

    buf[CYPRESS_BOOTLOADER_CMD_PACKET_SYNC_OFFSET] = CYPRESS_BOOTLOADER_CMD_HOST_SYNC;
    buf[CYPRESS_BOOTLOADER_CMD_PACKET_START_OFFSET] = CYPRESS_BOOTLOADER_CMD_PACKET_START;
    buf[CYPRESS_BOOTLOADER_CMD_PACKET_CMD_OFFSET] = cmd;
    buf[CYPRESS_BOOTLOADER_CMD_PACKET_DATA_LEN_LSB] = (data_len & LSB);      /* LSB */
    buf[CYPRESS_BOOTLOADER_CMD_PACKET_DATA_LEN_MSB] = (data_len & MSB >> 8); /* MSB */

    /* There is data to be sent to the controller */
    if (data_len != 0) {
        switch (cmd) {
        case CYPRESS_BOOTLOADER_CMD_INITIATE_BOOTLOAD:
        case CYPRESS_BOOTLOADER_CMD_PROGRAM_ROW:
        case CYPRESS_BOOTLOADER_CMD_VERIFY_ROW:
        case CYPRESS_BOOTLOADER_CMD_SEND_DATA:
            for (i = 0; i < data_len; i++) {
                buf[i + CYPRESS_BOOTLOADER_CMD_PACKET_DATA_OFFSET] = data[i];
            }
            break;
        default:
            break;
        }
    }

#if 0
    /* Break up the commands into 32 byte chunks */
    if ((cypress->use_send_data) && (CYPRESS_BOOTLOADER_CMD_PROGRAM_ROW == cmd)) {
        /* Data left is the row data minus the Array ID (1 byte) and
           Flash Row ID (2 bytes) */
        data_left = (data_len - 3);

        i = 0;

        do {
            memcpy (send_buf, buf[i + CYPRESS_BOOTLOADER_CMD_PACKET_DATA_OFFSET + 3], 32);
            i += 32;
        } while (data_left);
    }

#endif
    /* Consider the Start Packet, Command and the Data Length bytes 'data' for crc
       purposes, not sure why End Packet byte is not included in CRC */
    data_len += 4;

    crc = calculate_bootloader_crc(&(buf[1]), data_len);
    buf[len - 3] = (crc & LSB);      /* LSB */
    buf[len - 2] = (crc & MSB) >> 8; /* MSB */
    buf[len - 1] = CYPRESS_BOOTLOADER_CMD_PACKET_END;

    if (cypress->verbose > 5)
        cypress_bootloader_display_packet(cypress, buf, len, WRITE);

    /* Platform specific clear irq if exists. */
    if (cypress->platform.tp_clear_irq) {
        rc = cypress->platform.tp_clear_irq(cypress);
        if (EOK != rc) {
            mtouch_error(cypress->log_name, "Error performing platform-specific clear irq");
            error_memory ("Cypress_Touch: Error performing platform-specific clear irq");
        }
    } else
            InterruptUnmask(cypress->tp_intr, cypress->tp_iid);

    rc = cypress->i2c_funcs.write_reg(cypress->i2c_fd, addr, len, buf);
    if (0 != rc) {
        mtouch_error(cypress->log_name, "Failed to send Bootloader command %x", cmd);
        free (buf);
        return -1;
    }

    free (buf);
    return EOK;
}

int
cypress_bootloader_program_record(cypress_dev_t* cypress)
{
    uint8_t *buf;
    int data_len = 0;

    if (cypress->curr) {
        if (cypress->verbose > 1)
            mtouch_info (cypress->log_name, "Programming Row %x", cypress->curr->row_number);

        if (cypress->verbose)
            mtouch_info (cypress->log_name, "Firmware update %.2f%% complete", (((float)cypress->firmware_lines_complete/(float)cypress->firmware_line_count) * 100));

        data_len = (cypress->curr->record_length + sizeof (cypress->curr->array_id) + sizeof (cypress->curr->row_number));

        if (data_len < MIN_RECORD_DATA_LENGTH) {
            mtouch_error (cypress->log_name, "Record length + Array ID + Row Number is less than minimum amount (3)!!");
            return -1;
        }

        buf = (uint8_t *)calloc(data_len, sizeof (uint8_t));

        if (buf == NULL) {
            mtouch_error (cypress->log_name, "Failed to calloc space for program record command");
            return -1;
        }

        buf[0] = cypress->curr->array_id;
        buf[1] = (cypress->curr->row_number & 0xFF);
        buf[2] = ((cypress->curr->row_number >> 8) & 0xFF);
        memcpy (&(buf[3]), cypress->curr->data, cypress->curr->record_length);

#if 0
        if (cypress->use_send_data) {
            remaining_data = data_len;

            /* Breakup the command into 32byte chunks */
            for (i = 0; i < (data_len / 32); i++) {
                memcpy (&send_data_buf, buf[i * buf_size], buf_size);
                if (EOK != cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_SEND_DATA, 0, &send_data_buf, buf_size))
                    goto fail;
            }
        }
#endif

        /* Program row command */
        if (EOK != cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_PROGRAM_ROW, 0, buf, data_len))
            goto fail;

        free (buf);
    } else {
        cypress->program_record_state = FINISHED;
        cypress->firmware_lines_complete = 1;
        cypress->firmware_line_count = 0;
    }

    return EOK;

fail:
    if (buf != NULL)
        free (buf);

    cypress->firmware_lines_complete = 1;
    cypress->firmware_line_count = 0;

    return -1;
}

int
cypress_bootloader_update_firmware (cypress_dev_t* cypress)
{
    uint8_t initiate_bootload_data[] = {0xA5, 0x01, 0x02, 0x03, 0xFF, 0xFE, 0xFD, 0x5A};
    uint8_t *reply = NULL;
    int reply_len;
    uint8_t buf[3];
    static int toggle = 0;

    switch (cypress->firmware_update_state) {
    case INITIATE_BOOTLOAD:
        toggle = 0;

        /* Check Silicon ID */
        if (cypress->file_silicon_id != cypress->silicon_id) {
            mtouch_error (cypress->log_name, "Silicon ID in firmware CYACD file does not match controller Silicon ID", cypress->file_silicon_id, cypress->silicon_id);
            mtouch_error (cypress->log_name, "Found File ID: %x Chip reported ID: %x", cypress->file_silicon_id, cypress->silicon_id);
            goto fail;
        } else {
            if (cypress->verbose)
                mtouch_info (cypress->log_name, "Chip revision in firmware and CYACD file match");
        }

        if (cypress->verbose)
            mtouch_info (cypress->log_name, "Erasing flash...");

        /* Initiate bootload */
        if (EOK != cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_INITIATE_BOOTLOAD, 0, initiate_bootload_data, sizeof (initiate_bootload_data))) {
            mtouch_error (cypress->log_name, "Failed to send Initiate Bootload command");
            goto fail;
        }

        cypress->firmware_update_state = PROGRAM_RECORD;
        cypress->curr = cypress->head_record;
        break;
    case PROGRAM_RECORD:
        switch (cypress->program_record_state) {
        case WRITE:
            if ((reply_len = cypress_read_bootloader_reply (cypress, &reply)) == -1) {
                if (cypress->retry_attempts < cypress->retry_limit) {
                    mtouch_warn (cypress->log_name, "Retry attempt %d", cypress->retry_attempts);
                    cypress->retry_attempts++;
                    break;
                } else {
                    goto fail;
                }
            }

            cypress->retry_attempts = 0;

            if (toggle) {
                if ((quick_cksum(cypress->curr->data, cypress->curr->record_length)) != reply[4]) {
                    mtouch_error (cypress->log_name, "Flash Checksum does not match Program Data Checksum!");
                    free (reply);
                    goto fail;
                }
                cypress->curr = cypress->curr->next;
            }

            free (reply);

            if (EOK != cypress_bootloader_program_record(cypress)) {
                mtouch_error (cypress->log_name, "Failed to program records to touch controller");
                goto fail;
            }

            /* Check to see if we are finished */
            if (cypress->program_record_state == FINISHED) {
                mtouch_info (cypress->log_name, "Firmware updated succesfully");
                cleanup(cypress);
                free (cypress->firmware_file);
                mtouch_error (cypress->log_name, "Firmware update succesful, resetting ...");
                if (EOK == cypress_controller_reset(cypress)) {
                    /* Firmware update was successful */
                    cypress->driver_state = BOOTLOADER;
                    cypress->update_firmware = 0;
                    cypress_update_pps_status(cypress, "status::bootloader");
                    MsgReply(cypress->firmware_rcvid, EOK, NULL, 0);
                    cypress->firmware_rcvid = -1;
                } else {
                    goto fail;
                }

                cypress->firmware_update_state = OPEN_FILE;
            } else {
                cypress->program_record_state = VERIFY;
            }
            break;
        case VERIFY:
            if ((reply_len = cypress_read_bootloader_reply (cypress, &reply)) == -1) {
                if (cypress->retry_attempts < cypress->retry_limit) {
                    mtouch_warn (cypress->log_name, "Retry attempt %d", cypress->retry_attempts);
                    cypress->retry_attempts++;
                    break;
                } else {
                    goto fail;
                }
            }

            cypress->retry_attempts = 0;

            if (cypress_parse_bootlaoder_error(cypress, reply) != EOK) {
                free (reply);
                goto fail;
            }

            free (reply);

            buf[0] = cypress->curr->array_id;
            buf[1] = (cypress->curr->row_number & 0xFF);
            buf[2] = ((cypress->curr->row_number >> 8) & 0xFF);

            if (cypress->verbose > 1)
                mtouch_info (cypress->log_name, "Verifying Row: %x", cypress->curr->row_number);

            if (EOK != cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_VERIFY_ROW, 0, buf, (sizeof (cypress->curr->array_id) + sizeof (cypress->curr->row_number))))
                goto fail;

            cypress->program_record_state = WRITE;
            cypress->firmware_lines_complete++;
            toggle = 1;
            break;
        default:
            mtouch_error (cypress->log_name, "Unknown state for program record: %d", cypress->program_record_state);
            goto fail;
            break;
        }
        break;
    default:
        mtouch_error (cypress->log_name, "Unknown state for firmware upgrade: %d", cypress->firmware_update_state);
        goto fail;
        break;
    }

    return EOK;

fail:
    mtouch_error (cypress->log_name, "Firmware update failed");
    error_memory("Cypress_Touch: Firmware update failed");
    MsgReply(cypress->firmware_rcvid, -1, NULL, 0);
    cypress->firmware_rcvid = -1;
    cypress->update_firmware = 0;
	cleanup(cypress);
    mtouch_error (cypress->log_name, "Firmware update failed, resetting ...");
    if (EOK != cypress_controller_reset(cypress)) {
        mtouch_error (cypress->log_name, "Failed to reset the controller");
        error_memory ("Cypress_Touch: Failed to reset the controller");
    }

    return -1;
}

int
bootloader_check(cypress_dev_t *cypress)
{
    uint8_t reg[2];
    int rc;

    /* Check to see if we are in boot loader mode */
    if ((rc = cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, 2, reg)) == 0) {
        mtouch_info (cypress->log_name, "cypress mode : %d %d", reg[HST_MODE], reg[RESET_DETECT]);
        if ((reg[HST_MODE] == 1) || (reg[RESET_DETECT] != 0)) {
            if (cypress->verbose > 4)
                mtouch_info (cypress->log_name, "Controller is in Bootloader mode");
            return EOK;
        } else {
            /* ERROR! */
            mtouch_error (cypress->log_name, "Bootloader check, cypress controller in wrong state, resetting");
            if (EOK != cypress_controller_reset(cypress)) {
                mtouch_error (cypress->log_name, "Failed to reset the controller");
            }
            return -1;
        }
    } else {
        rc = -1;
        mtouch_error(cypress->log_name, "failed to read mode: i2c failed %d", rc);
    }

    return rc;
}

int
bootloader_state_machine(cypress_dev_t *cypress)
{
    uint8_t *reply = NULL;
    int reply_len = 0;
    static int reply_expected = 0;

    switch (cypress->bootloader_state) {
    case IDLE:
        if (EOK != bootloader_check(cypress))
            return -1;

        if (EOK != cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_ENTER_ACTIVE_STATE, 0, NULL, 0))
            return -1;
        mtouch_error (cypress->log_name, "Set bootloader to active state");
        cypress->bootloader_state = ACTIVE;
        reply_expected = 1;

        /* Start watchdog, just in case controller gets stuck */
        cypress_set_boot_watchdog_timer(cypress, cypress->boot_watchdog_sleep);

        break;
    case ACTIVE:
        /* Stop watchdog */
        cypress_set_boot_watchdog_timer(cypress, 0);

        if (EOK != bootloader_check(cypress))
            return -1;

        if (reply_expected) {
            /* Read bootloader reply for the silicon ID incase things went bad and we are waiting for a firmware update */
            if ((reply_len = cypress_read_bootloader_reply (cypress, &reply)) == -1) {
                if (cypress->retry_attempts < cypress->retry_limit) {
                    mtouch_warn (cypress->log_name, "Unable to read bootloader reply.  Retry attempt %d", cypress->retry_attempts);
                    cypress->retry_attempts++;

                    /* Start watchdog, just in case controller gets stuck */
                    cypress_set_boot_watchdog_timer(cypress, cypress->boot_watchdog_sleep);
                    return EOK;
                } else {
                    return -1;
                }
            } else {
                cypress->retry_attempts = 0;
            }

            if (cypress->verbose > 6)
                cypress_bootloader_display_packet(cypress, reply, reply_len, READ);

            cypress->silicon_gen = cypress_decypher_family_gen (cypress, reply[4]);//((reply[5] << 8) | reply[4]);
            cypress->silicon_id = ((reply[7] << 8) | reply[6]);
            cypress->chip_revision = reply[8];
            reply_expected = 0;

            free (reply);
        }

        if (1 == cypress->update_firmware) {
            cypress_bootloader_update_firmware (cypress);
        } else {
            if (EOK != cypress_bootloader_cmd(cypress, CYPRESS_BOOTLOADER_CMD_EXIT_CRC, 0, NULL, 0)) {
                mtouch_error(cypress->log_name, "failed to execute exit bootloader cmd: i2c failed");
                return -1;
            }
            mtouch_error (cypress->log_name, "Exit bootloader command properly sent");

            cypress->bootloader_state = APPLICATION_RUNNING;
            cypress->driver_state = SYSTEM_INFORMATION;
            cypress_update_pps_status(cypress, "status::system_information");
        }

        /* Start watchdog, just in case controller gets stuck */
        cypress_set_boot_watchdog_timer(cypress, cypress->boot_watchdog_sleep);
        break;
    case APPLICATION_RUNNING:
        mtouch_error (cypress->log_name, "%s: should never see this.  State %d", __FUNCTION__, cypress->bootloader_state);
        break;
    case WAITING_UPDATE:
        cypress_set_boot_watchdog_timer(cypress, 0);
        mtouch_info (cypress->log_name, "Driver is waiting for a firmware update...");
        cypress_set_boot_watchdog_timer(cypress, cypress->boot_watchdog_sleep);
        break;
    case CLEARING_ERROR:
        cypress_bootloader_clear_error_pin(cypress);
        break;
    default:
        mtouch_error (cypress->log_name, "Bootloader is in an unknown state: %d", cypress->bootloader_state);
        error_memory ("Cypress_Touch: Bootloader is in an unknown state: %d", cypress->bootloader_state);
        return -1;
    }

    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/bootloader.c $ $Rev: 886397 $")
#endif
